// Popup script for PatternGuard

document.addEventListener('DOMContentLoaded', async () => {
  // Get current tab
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  
  if (!tab?.id) {
    showEmptyState('Unable to access current tab');
    return;
  }

  // Request stats from content script
  try {
    const response = await chrome.tabs.sendMessage(tab.id, { type: 'GET_STATS' });
    
    if (response) {
      displayStats(response);
      
      // Set toggle state
      const toggle = document.getElementById('highlightToggle');
      toggle.checked = response.highlightEnabled !== false;
    } else {
      showEmptyState('No data available for this page');
    }
  } catch (error) {
    console.error('Error getting stats:', error);
    showEmptyState('No patterns detected on this page');
  }

  // Highlight toggle
  document.getElementById('highlightToggle').addEventListener('change', async (e) => {
    try {
      const response = await chrome.tabs.sendMessage(tab.id, {
        type: 'TOGGLE_HIGHLIGHT',
        enabled: e.target.checked
      });
      console.log('Toggle response:', response);
    } catch (error) {
      console.error('Error toggling highlight:', error);
    }
  });

  // Reset button
  document.getElementById('resetBtn').addEventListener('click', async (e) => {
    e.preventDefault();
    
    try {
      await chrome.storage.local.set({
        detectedPatterns: [],
        lastUpdate: Date.now()
      });
      
      // Refresh display
      document.getElementById('detectionCount').textContent = '0';
      document.getElementById('patternsContainer').innerHTML = 
        '<div class="empty-state">Stats reset successfully</div>';
      
      // Reload the page after a short delay
      setTimeout(() => window.location.reload(), 1000);
    } catch (error) {
      console.error('Error resetting stats:', error);
    }
  });
});

function displayStats(data) {
  const { count, patterns } = data;
  
  // Update count
  document.getElementById('detectionCount').textContent = count;
  
  // Display patterns
  const container = document.getElementById('patternsContainer');
  
  if (patterns && patterns.length > 0) {
    // Show most recent 5 patterns
    const recentPatterns = patterns.slice(-5).reverse();
    
    container.innerHTML = recentPatterns.map(pattern => `
      <div class="pattern-item">
        <div class="pattern-name">${pattern.name}</div>
        <div class="pattern-text">${escapeHtml(pattern.text)}</div>
      </div>
    `).join('');
  } else {
    showEmptyState('No patterns detected yet');
  }
}

function showEmptyState(message) {
  document.getElementById('detectionCount').textContent = '0';
  document.getElementById('patternsContainer').innerHTML = 
    `<div class="empty-state">${message}</div>`;
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}
