// Content script for PatternGuard
// Detects dark patterns on web pages using MutationObserver

// Pattern detection rules
const PATTERNS = {
  countdownTimer: {
    name: 'Countdown Timer',
    keywords: [
      // More specific patterns that combine time with urgency
      /\d+:\d+:\d+ (left|remaining)/i,
      /(ends?|expires?|closes?) in \d+:\d+/i,
      /\d+ (hours?|minutes?|seconds?|days?) (left|remaining)/i,
      /(hurry|act now|don't wait).*(limited time|hours? left|minutes? left)/i,
      /sale ends in \d+/i,
      /offer expires? in \d+/i
    ],
    // Patterns to exclude (false positives)
    excludePatterns: [
      /last updated/i,
      /posted (at|on)/i,
      /published (at|on)/i,
      /duration:/i,
      /video length/i,
      /runtime:/i,
      /starts? at/i,
      /uploaded/i,
      /timestamp/i,
      /\d{4}-\d{2}-\d{2}/,  // Date format YYYY-MM-DD
      /jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec/i  // Month names
    ],
    severity: 'medium',
    customCheck: checkCountdownTimer
  },
  falseScarcity: {
    name: 'False Scarcity',
    keywords: [
      /only \d+ left/i,
      /\d+ (items?|spots?|seats?) (left|remaining)/i,
      /low stock/i,
      /selling fast/i,
      /almost gone/i
    ],
    severity: 'high'
  },
  fakeUrgency: {
    name: 'Fake Urgency',
    keywords: [
      /don't miss out/i,
      /last chance/i,
      /act now/i,
      /limited offer/i,
      /while supplies last/i
    ],
    severity: 'medium'
  },
  hiddenCosts: {
    name: 'Hidden Costs',
    keywords: [
      /\+ fees?/i,
      /additional charges?/i,
      /taxes? (and|&) fees?/i,
      /service charges?/i
    ],
    severity: 'high'
  }
};

// Statistics
let detectedPatterns = [];
let highlightEnabled = true;

// Custom check function for countdown timers
function checkCountdownTimer(element, text) {
  // Check for exclude patterns first
  const excludePatterns = PATTERNS.countdownTimer.excludePatterns;
  for (const excludePattern of excludePatterns) {
    if (excludePattern.test(text)) {
      return false; // Skip this element, it's a false positive
    }
  }
  
  // If we have a standalone time format (HH:MM:SS), it needs urgency context
  const standaloneTimeMatch = text.match(/\d+:\d+:\d+/);
  if (standaloneTimeMatch) {
    // Check if there's urgency context nearby
    const hasUrgencyContext = /(left|remaining|ends?|expires?|hurry|limited|offer|sale|deal)/i.test(text);
    if (!hasUrgencyContext) {
      return false;
    }
  }
  
  // Check against positive patterns
  const keywords = PATTERNS.countdownTimer.keywords;
  for (const keyword of keywords) {
    if (keyword.test(text)) {
      return true;
    }
  }
  
  return false;
}

// Initialize
function init() {
  console.log('[PatternGuard] Initializing...');
  
  // Load user preferences
  chrome.storage.local.get(['highlightEnabled'], (result) => {
    highlightEnabled = result.highlightEnabled !== false; // Default to true
    
    // Scan existing content
    scanPage();
    
    // Watch for dynamic content changes
    observePageChanges();
    
    // Send initial stats
    updateBadge();
  });
}

// Scan the entire page for dark patterns
function scanPage() {
  const allElements = document.querySelectorAll('body *');
  
  allElements.forEach(element => {
    // Skip script and style elements
    if (element.tagName === 'SCRIPT' || element.tagName === 'STYLE') {
      return;
    }
    
    const text = element.textContent?.trim();
    if (!text || text.length > 200) return; // Skip very long text blocks
    
    checkElement(element, text);
  });
}

// Check if an element contains dark patterns
function checkElement(element, text) {
  for (const [patternType, pattern] of Object.entries(PATTERNS)) {
    // Use custom check function if available
    if (pattern.customCheck) {
      if (pattern.customCheck(element, text)) {
        highlightElement(element, pattern.name, pattern.severity);
        
        // Record the detection
        detectedPatterns.push({
          type: patternType,
          name: pattern.name,
          text: text.substring(0, 100),
          timestamp: Date.now()
        });
        
        updateBadge();
        break;
      }
    } else {
      // Standard keyword matching
      for (const keyword of pattern.keywords) {
        if (keyword.test(text)) {
          highlightElement(element, pattern.name, pattern.severity);
          
          // Record the detection
          detectedPatterns.push({
            type: patternType,
            name: pattern.name,
            text: text.substring(0, 100),
            timestamp: Date.now()
          });
          
          updateBadge();
          break;
        }
      }
    }
  }
}

// Highlight detected elements with a subtle overlay
function highlightElement(element, patternName, severity) {
  // Avoid highlighting already highlighted elements
  if (element.hasAttribute('data-pattern-guard')) {
    return;
  }
  
  element.setAttribute('data-pattern-guard', patternName);
  element.setAttribute('data-pattern-severity', severity);
  
  // Only apply visual highlighting if enabled
  if (highlightEnabled) {
    applyHighlight(element, patternName, severity);
  }
}

// Apply visual highlighting
function applyHighlight(element, patternName, severity) {
  element.classList.add('pattern-guard-highlight');
  element.classList.add(`pattern-guard-${severity}`);
  
  // Create tooltip
  const tooltip = document.createElement('div');
  tooltip.className = 'pattern-guard-tooltip';
  tooltip.textContent = `⚠️ ${patternName}`;
  element.appendChild(tooltip);
  
  // Create marker badge
  const marker = document.createElement('div');
  marker.className = 'pattern-guard-marker';
  marker.textContent = '!';
  marker.title = patternName;
  element.style.position = element.style.position || 'relative';
  element.appendChild(marker);
}

// Remove visual highlighting
function removeHighlight(element) {
  element.classList.remove('pattern-guard-highlight', 'pattern-guard-high', 'pattern-guard-medium');
  
  // Remove tooltip and marker
  const tooltip = element.querySelector('.pattern-guard-tooltip');
  const marker = element.querySelector('.pattern-guard-marker');
  if (tooltip) tooltip.remove();
  if (marker) marker.remove();
}

// Toggle highlighting for all detected patterns
function toggleHighlighting(enabled) {
  highlightEnabled = enabled;
  
  // Save preference
  chrome.storage.local.set({ highlightEnabled: enabled });
  
  // Apply or remove highlighting from all detected elements
  const highlightedElements = document.querySelectorAll('[data-pattern-guard]');
  highlightedElements.forEach(element => {
    const patternName = element.getAttribute('data-pattern-guard');
    const severity = element.getAttribute('data-pattern-severity');
    
    if (enabled) {
      applyHighlight(element, patternName, severity);
    } else {
      removeHighlight(element);
    }
  });
}

// Observe DOM changes for dynamically loaded content
function observePageChanges() {
  const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      mutation.addedNodes.forEach((node) => {
        if (node.nodeType === Node.ELEMENT_NODE) {
          const text = node.textContent?.trim();
          if (text && text.length <= 200) {
            checkElement(node, text);
          }
          
          // Check child elements
          const children = node.querySelectorAll('*');
          children.forEach(child => {
            const childText = child.textContent?.trim();
            if (childText && childText.length <= 200) {
              checkElement(child, childText);
            }
          });
        }
      });
    });
  });
  
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
  
  console.log('[PatternGuard] MutationObserver active');
}

// Update extension badge with detection count
function updateBadge() {
  chrome.runtime.sendMessage({
    type: 'UPDATE_BADGE',
    count: detectedPatterns.length
  });
  
  // Store stats in local storage
  chrome.storage.local.set({
    detectedPatterns: detectedPatterns,
    lastUpdate: Date.now()
  });
}

// Listen for messages from popup or background
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.type === 'GET_STATS') {
    sendResponse({
      count: detectedPatterns.length,
      patterns: detectedPatterns,
      highlightEnabled: highlightEnabled
    });
  } else if (request.type === 'TOGGLE_HIGHLIGHT') {
    toggleHighlighting(request.enabled);
    sendResponse({ success: true, enabled: highlightEnabled });
  }
});

// Start detection when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}

console.log('[PatternGuard] Content script loaded');
