// Background service worker for PatternGuard
// Handles badge updates and API communication

console.log('[PatternGuard] Background service worker started');

// Listen for messages from content scripts
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.type === 'UPDATE_BADGE') {
    updateBadge(sender.tab?.id, request.count);
  }
});

// Update extension badge
function updateBadge(tabId, count) {
  if (!tabId) return;
  
  const text = count > 0 ? count.toString() : '';
  const color = count > 5 ? '#DC2626' : count > 0 ? '#F59E0B' : '#10B981';
  
  chrome.action.setBadgeText({
    text: text,
    tabId: tabId
  });
  
  chrome.action.setBadgeBackgroundColor({
    color: color,
    tabId: tabId
  });
}

// Optional: Send analysis to external API
async function analyzeWithAPI(patterns) {
  try {
    const response = await fetch('https://api.yourvps.com/analyze', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        patterns: patterns,
        timestamp: Date.now()
      })
    });
    
    if (response.ok) {
      const data = await response.json();
      console.log('[PatternGuard] API analysis:', data);
      return data;
    }
  } catch (error) {
    console.error('[PatternGuard] API error:', error);
  }
  
  return null;
}

// Reset badge on tab updates
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'loading') {
    chrome.action.setBadgeText({
      text: '',
      tabId: tabId
    });
  }
});

// Initialize on install
chrome.runtime.onInstalled.addListener(() => {
  console.log('[PatternGuard] Extension installed');
  
  // Set default storage
  chrome.storage.local.set({
    totalDetections: 0,
    lastReset: Date.now()
  });
});
